<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Role;
// use Spatie\Permission\Models\Role;
use Spatie\Permission\Models\Permission;
use App\Models\User;
use Illuminate\Support\Facades\DB;
use App\Models\RoleDataScope;
use App\Models\Department;

class RoleController extends Controller
{
  public function create()
{
    // Predefined modules and their relevant permissions
    $modules = [
        'profile' => ['profile.edit', 'profile.update', 'profile.destroy'],
        'room' => ['room.index', 'add-room', 'update-room', 'delete-room'],
        'tag' => ['tag.index', 'add-tag', 'update-tag', 'delete-tag'],
        'category' => ['category.index', 'add-category', 'update-category', 'delete-category'],
        'subcategory' => ['subcategory', 'add-subcategory', 'update-subcategory', 'delete-subcategory', 'toggle-subcategory-status'],
        'product' => ['products', 'edit-product', 'add-product', 'delete-product'],
        'banner' => ['banner', 'get-banner', 'add-banner', 'update-banner', 'banner.status'],
        'coupon' => ['coupons', 'get-coupons', 'add-coupon', 'coupon.update', 'coupon.status'],
        'stock' => ['stock.index', 'stock.list', 'stock.view', 'stock.edit'],
        'location' => ['locations', 'get-locations', 'add-locations', 'update-locations', 'delete-locations'],
        'roles' => ['roles.index', 'roles.create', 'roles.edit', 'roles.update', 'roles.destroy'],
        'users' => ['users.index', 'users.store', 'users.edit', 'users.update', 'users.destroy']
    ];

    // Fetch all permissions
    $allPermissions = Permission::all()->pluck('name')->toArray();

    // Group permissions according to modules
    $permissions = [];
    foreach ($modules as $module => $perms) {
        $permissions[$module] = array_values(array_intersect($allPermissions, $perms));
    }

    // For create, no permissions are assigned yet
    $rolePermissions = [];

    return view('admin.roles.create', compact('permissions', 'rolePermissions'));
}

public function store(Request $request)
{
    $request->validate([
        'role_name'       => 'required|string|unique:roles,name',
        'permissions'     => 'array',
        'department_ids'  => 'array',
        'department_ids.*'=> 'exists:departments,id',
    ]);

    DB::transaction(function () use ($request) {

        /** -----------------------------
         * 1️⃣ Create Role
         * ---------------------------- */
        $role = Role::create([
            'name'       => $request->role_name,
            'guard_name' => 'web',
        ]);

        /** -----------------------------
         * 2️⃣ Attach Permissions
         * ---------------------------- */
        if ($request->filled('permissions')) {
            $role->syncPermissions($request->permissions);
        }

        /** -----------------------------
         * 3️⃣ Attach Departments (Pivot)
         * ---------------------------- */
        if ($request->filled('department_ids')) {
            $role->departments()->sync($request->department_ids);
        }
    });

    return redirect()
        ->route('roles.index')
        ->with('success', 'Role created successfully');
}



    public function index()
    {
        $roles = Role::withCount('permissions')->get();
        return view('admin.roles.index', compact('roles'));
    }



    public function destroy(Role $role)
    {
        if ($role->name === 'Super Admin') {
            return response()->json([
                'status' => false,
                'message' => 'Admin role cannot be deleted'
            ], 403);
        }

        $role->delete();

        return response()->json([
            'status' => true,
            'message' => 'Role deleted successfully'
        ]);
    }

  public function edit(Role $role)
{
    // -----------------------------
    // Modules & permissions (UNCHANGED)
    // -----------------------------
    $modules = [
        'profile' => [
            'view'   => 'profile.index',
            'edit'   => 'profile.edit',
            'delete' => 'profile.destroy',
        ],
        'room' => [
            'view'   => 'room.index',
            'add'    => 'add-room',
            'edit'   => 'update-room',
            'delete' => 'delete-room',
        ],
        'tag' => [
            'view'   => 'tag.index',
            'add'    => 'add-tag',
            'edit'   => 'update-tag',
            'delete' => 'delete-tag',
        ],
        'category' => [
            'view'   => 'category.index',
            'add'    => 'add-category',
            'edit'   => 'update-category',
            'delete' => 'delete-category',
        ],
        'subcategory' => [
            'view'   => 'subcategory',
            'add'    => 'add-subcategory',
            'edit'   => 'update-subcategory',
            'delete' => 'delete-subcategory',
            'toggle' => 'toggle-subcategory-status'
        ],
        'product' => [
            'view'   => 'products',
            'add'    => 'add-product',
            'edit'   => 'edit-product',
            'delete' => 'delete-product'
        ],
        'banner' => [
            'view'   => 'banner',
            'add'    => 'add-banner',
            'edit'   => 'update-banner',
            'status' => 'banner.status'
        ],
        'coupon' => [
            'view'   => 'coupons',
            'add'    => 'add-coupon',
            'edit'   => 'coupon.update',
            'status' => 'coupon.status'
        ],
        'stock' => [
            'view' => 'stock.index',
            'list' => 'stock.list',
            'edit' => 'stock.edit'
        ],
        'location' => [
            'view'   => 'locations',
            'add'    => 'add-locations',
            'edit'   => 'update-locations',
            'delete' => 'delete-locations'
        ],
        'roles' => [
            'view'   => 'roles.index',
            'add'    => 'roles.create',
            'edit'   => 'roles.edit',
            'delete' => 'roles.destroy'
        ],
        'users' => [
            'view'   => 'users.index',
            'add'    => 'users.store',
            'edit'   => 'users.edit',
            'delete' => 'users.destroy'
        ],
    ];

    // -----------------------------
    // Build permission matrix
    // -----------------------------
    $permissions = [];
    $allPermissions = Permission::pluck('name')->toArray();

    foreach ($modules as $module => $actions) {
        foreach ($actions as $action => $permName) {
            if (in_array($permName, $allPermissions)) {
                $permissions[$module][$action] = $permName;
            }
        }
    }

    // -----------------------------
    // Role current data
    // -----------------------------
    $rolePermissions = $role->permissions->pluck('name')->toArray();

    $departments = Department::where('is_active', 1)->get();
    $selectedDepartments = $role->departments->pluck('id')->toArray();
$roleScopes = $role->scopes ?? []; // saved JSON / array
    return view('admin.roles.edit', compact(
        'role',
        'permissions',
        'rolePermissions',
        'departments',
        'selectedDepartments',
        'roleScopes'
    ));
}




public function update(Request $request, Role $role)
{
    // Optional lock check
    // if ($role->name === 'Super Admin') { ... }

    $request->validate([
        'role_name'        => 'required|string|unique:roles,name,' . $role->id,
        'permissions'      => 'array',
        'permissions.*'    => 'exists:permissions,name',
        'department_ids'   => 'array',
        'department_ids.*' => 'exists:departments,id',
    ]);

    DB::transaction(function () use ($request, $role) {

        // -----------------------------
        // Update role
        // -----------------------------
        $role->update([
            'name' => $request->role_name
        ]);

        // -----------------------------
        // Sync permissions
        // -----------------------------
        $role->syncPermissions($request->permissions ?? []);

        // -----------------------------
        // Sync departments (pivot)
        // -----------------------------
        if ($request->has('department_ids')) {
            $role->departments()->sync($request->department_ids);
        }
    });

    return response()->json([
        'status'  => true,
        'message' => 'Role updated successfully'
    ]);
}





    public function deleteSelectedPermissions(Request $request)
    {
        // Predefined list of permissions that are protected (cannot be deleted)
        $permissionsToDelete  = [
            'products.step.one',
            'product.step2.store',
            'product.images.store',
            'stocks.store',
            'products.step.one.update',
            'products.step.two.update',
            'products.step.three.update',
            'products.step.four.update',
            'products.image.delete',
        ];

        DB::transaction(function () use ($permissionsToDelete) {
            foreach ($permissionsToDelete as $permName) {
                $permission = Permission::where('name', $permName)->first();
                if ($permission) {
                    // Remove permission from roles
                    foreach ($permission->roles as $role) {
                        $role->revokePermissionTo($permission);
                    }

                    // Remove permission from users
                    foreach ($permission->users as $user) {
                        $user->revokePermissionTo($permission);
                    }

                    // Delete the permission
                    $permission->delete();
                }
            }
        });

        return response()->json([
            'status' => true,
            'message' => 'Selected permissions deleted successfully.'
        ]);
    }
}
